#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "smv_Types.h"
#include "smv_Disc.h"
#include "smv_Parse.h"
#include "smv_Object.h"
#include "smv_ChunkIDs.h"

/*
** Parsing Routines
*/

/*
** Each parse module creates its own data block (if appropriate)
** and passes it back to the caller, if successful.
** Should any of its children fail to initialise correctly, this
** makes it very easy to release all the memory from the root
** object, so a NULL ptr can be passed back to the main load
** routine
*/

BOOL	mv_ParseVertexList(FILE *file_ptr, MV_MODEL *pMesh, tLONG size)
{
tWORD i;
BOOL rt;
tLONG bytes_read = 0;

	bytes_read += sizeof(tWORD)+sizeof(tLONG);		/* we've got the header */

	if (mv_ReadWord(file_ptr, &pMesh->iNumVertices) == FALSE)	return FALSE;
	bytes_read += 2;

	pMesh->pVertexList = (MVERTEX *)calloc(pMesh->iNumVertices, sizeof(MVERTEX));
	if (!pMesh->pVertexList)
		return FALSE;

	rt = TRUE;
	for(i=0;i<pMesh->iNumVertices;i++)
		{
		rt &= mv_ReadFloat(file_ptr, &pMesh->pVertexList[i].x);
		rt &= mv_ReadFloat(file_ptr, &pMesh->pVertexList[i].y);
		rt &= mv_ReadFloat(file_ptr, &pMesh->pVertexList[i].z);
		}

	bytes_read += 3*sizeof(tLONG) * pMesh->iNumVertices;	

	if (!rt)  	/* something failed - release our memory, and move on */
		free(pMesh->pVertexList);

	/* Ignore any other data in this chunk */
	mv_SkipBytes(file_ptr, size - bytes_read);

	return rt;
}

BOOL	mv_ParseFaceList(FILE *file_ptr, MV_MODEL *pMesh, tLONG size)
{
tWORD i;
BOOL rt;
tWORD tmp;
tLONG bytes_read = 0;

	bytes_read += sizeof(tWORD)+sizeof(tLONG);		/* we've got the header */

	if (mv_ReadWord(file_ptr, &pMesh->iNumFaces) == FALSE)	return FALSE;
	bytes_read += 2;

	pMesh->pFaceList = (MFACE *)calloc(pMesh->iNumFaces, sizeof(MFACE));
	if (!pMesh->pFaceList)
		return FALSE;
	
	rt = TRUE;
	for(i=0;i<pMesh->iNumFaces;i++)
		{
		rt &= mv_ReadWord(file_ptr, &pMesh->pFaceList[i].v1);
		rt &= mv_ReadWord(file_ptr, &pMesh->pFaceList[i].v2);
		rt &= mv_ReadWord(file_ptr, &pMesh->pFaceList[i].v3);
		rt &= mv_ReadWord(file_ptr, &tmp); /* flags: skip them for now... */
		}                         	
	bytes_read +=  pMesh->iNumFaces * 4 * sizeof(tWORD);

	if (!rt)  	/* something failed - release our memory, and move on */
		free(pMesh->pFaceList);
                 	
	/* Ignore any other data in this chunk */
	mv_SkipBytes(file_ptr, size - bytes_read);

	return rt;
}

BOOL mv_ParsePolygonData(FILE *file_ptr, tLONG block_size, MV_MODEL *pMesh)
{
tWORD id;
tLONG size;
tLONG end_of_block;

	end_of_block = ftell(file_ptr) + block_size; 	/* where the block should end... */
	end_of_block -= sizeof(tWORD)+sizeof(tLONG);	/*...ignoring the header */

	do
		{
		mv_ReadChunk(file_ptr, &id, &size);

		switch(id)
			{
			case	SMV_VERTEXLIST:
						mv_ParseVertexList(file_ptr, pMesh, size);	
						break;
			case	SMV_FACELIST:
						mv_ParseFaceList(file_ptr, pMesh, size);
						break;
			default:
						mv_SkipChunk(file_ptr, size);
			}
		}
	while(ftell(file_ptr) < end_of_block);
	
	return TRUE;
}


BOOL mv_ParseObjectBlock(FILE *file_ptr, tLONG block_size, MV_OBJECT *pObj, MV_MODEL *pMesh)
{
tWORD id;
tLONG size;
tLONG end_of_block;

	end_of_block = ftell(file_ptr) + block_size; 	/* where the block should end... */
	end_of_block -= sizeof(tWORD)+sizeof(tLONG);	/*...ignoring the header */

	mv_ReadString(file_ptr, pObj->szName, sizeof(pObj->szName));	

	do
		{
		if (mv_ReadChunk(file_ptr, &id, &size) == FALSE)
			return FALSE;

		switch(id)
			{
			case	SMV_POLYGONDATA:
						mv_ParsePolygonData(file_ptr, size, pMesh);
						break;
			case	SMV_MESHLIGHT:
						mv_SkipChunk(file_ptr, size);
						break;
			case	SMV_MESHCAMERA:
						mv_SkipChunk(file_ptr, size);
						break;
			default:
						mv_SkipChunk(file_ptr, size);
			}
		}
	while(ftell(file_ptr) < end_of_block);
	
	return TRUE;
}

MV_MODEL *mv_ParseMeshData(FILE *file_ptr, tLONG mesh_size, MV_OBJECT *pObj)
{
tWORD id;
tLONG size;
tLONG end_of_block;
MV_MODEL	*pMesh;

	end_of_block = ftell(file_ptr) + mesh_size; 	/* where the block should end... */
	end_of_block -= sizeof(tWORD)+sizeof(tLONG);	/*...ignoring the header */

	pMesh = (MV_MODEL *)malloc(sizeof(MV_MODEL));
	if (!pMesh)
		return (MV_MODEL *)0;

	pMesh->pVertexList = 0;
	pMesh->pFaceList = 0;

	do
		{
		mv_ReadChunk(file_ptr, &id, &size);

		switch(id)
			{
			case	SMV_OBJECTDESCRIPTION:
						mv_ParseObjectBlock(file_ptr, size, pObj, pMesh);
						break;
			default:
						mv_SkipChunk(file_ptr, size);
			}
		}
	while(ftell(file_ptr) < end_of_block);
	
	return pMesh;
}


MV_OBJECT *mv_ParseMainBlock(FILE *file_ptr)
{
tWORD id;
tLONG size;
tLONG end_of_block;
MV_OBJECT *pObj;
MV_MODEL	*pPrevMesh;

	if (mv_ReadChunk(file_ptr, &id, &size) == FALSE)	
		return (MV_OBJECT *)0;

	end_of_block = ftell(file_ptr) + size; 	/* where the block should end... */
	end_of_block -= sizeof(tWORD)+sizeof(tLONG);	/*...ignoring the header */

	/* There's only one type of block at the root of the hierachy */
	/* Use names for the chunk IDs, as it tends towards self-documenting code */
	if (id != SMV_MAINBLOCK)	
		return (MV_OBJECT *)0;

	pObj = Obj_CreateObject();
	if (!pObj)
		return 0;

	/* Inside the main block there can be blocks containing mesh data
		keyframe (animation) data as so on. We're only interested in meshes */
	do
		{
   	mv_ReadChunk(file_ptr, &id, &size);

		switch(id)
			{
			case	SMV_MESH_DATA:
							/* a linked list for */
							pPrevMesh = pObj->pMesh;
							if ((pObj->pMesh = mv_ParseMeshData(file_ptr, size, pObj)))
								pObj->pMesh->pNext = pPrevMesh;
							break;

			case	SMV_KEYFRAMES:		/* we don't handle these */
			default:
          				mv_SkipChunk(file_ptr, size);
			}
		}
	while(ftell(file_ptr) < end_of_block);

	return pObj;
}

MV_OBJECT *mv_ParseFile(const char *pName)
{
FILE *file_ptr;

	if ((file_ptr = fopen(pName, "rb")))
		{
		MV_OBJECT *pObj;

	   pObj = mv_ParseMainBlock(file_ptr);
		fclose(file_ptr);
		return pObj;
		}
	return (MV_OBJECT *)0;
}
